/**
 *   RexxScriptEngine
 *   Copyright 2010 Stani Ryabenkiy
 *
 *   Licensed under the Apache License, Version 2.0 (the "License");
 *   you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and
 *   limitations under the License.
 *   
 */
package com.sun.script.rexx;

import java.util.ArrayList;
import java.util.List;

import javax.script.ScriptEngine;
import javax.script.ScriptEngineFactory;

/**
 * A JSR-223 compliant ScriptEngine for BSF 2.4 / BSF4ooRexx / ooRexx.
 * 
 * Loading the bindings each time a script is evaluated might not be the most efficient solution, but it does work. I'm not sure
 * that there is a more elegant solution given that bindings can come from diverse ScriptContext/Bindings sources - with the result
 * that the exact set of bindings to be loaded is not determined until the evaluation methods are called.
 * 
 * @author stani
 *
 */
public class RexxScriptEngineFactory implements ScriptEngineFactory {
	
	@Override
	public String getEngineName() {
		return "BSF4ooRexx Script Engine";
	}

	@Override
	public String getEngineVersion() {
		return "1.0";
	}

	@Override
	public List<String> getExtensions() {
		List<String> ext = new ArrayList<String>();
		
		ext.add("rex");
		ext.add("rxs");
		
		return ext;
	}

	@Override
	public String getLanguageName() {
		return "rexx";
	}

	@Override
	public String getLanguageVersion() {
		return "4.0.0";
	}

	@Override
	public String getMethodCallSyntax(String obj, String m, String... args) {
		
		String call = obj + "~" + m + "(";
		 for (int i = 0; i < args.length; i++) {
	          call += args[i];
	          if (i == args.length - 1) {
	        	  call += ")";
	          } else {
	        	  call += ",";
	          }
	      }
		return call;
	}

	@Override
	public List<String> getMimeTypes() {
		List<String> mimes = new ArrayList<String>();
		
		mimes.add("text/x-script.rexx");
		
		return mimes;
	}

	@Override
	public List<String> getNames() {
		List<String> names = new ArrayList<String>();
		
		names.add("rexx");
		names.add("oorexx");
		names.add("bsf4oorexx");
		
		return names;
	}

	@Override
	public String getOutputStatement(String toDisplay) {
		return "say \"" + toDisplay + "\"";
	}

	@Override
	public Object getParameter(String key) {
		if (key.equals(ScriptEngine.ENGINE)) {
            return getEngineName();
        } else if (key.equals(ScriptEngine.ENGINE_VERSION)) {
            return getEngineVersion();
        } else if (key.equals(ScriptEngine.NAME)) {
            return getEngineName();
        } else if (key.equals(ScriptEngine.LANGUAGE)) {
            return getLanguageName();
        } else if (key.equals(ScriptEngine.LANGUAGE_VERSION)) {
            return getLanguageVersion();
        } else if (key.equals("THREADING")) {
            return null;
        } else {
            return null;
        }
	}

	@Override
	public String getProgram(String... statements) {
		
		String ret = "";
		
		for (String statement : statements) {
			ret += statement + "\n";
		}
		
		return ret;
	}

	@Override
	public ScriptEngine getScriptEngine() {
		return new RexxScriptEngine(this);
	}

}
